const API_BASE = 'https://api.point.monster/api';

// インストール時の処理
chrome.runtime.onInstalled.addListener(() => {
    chrome.storage.local.get(['userId'], (result) => {
        // IDがなければ作る（Google認証前の一時的なIDとして機能）
        if (!result.userId) {
            chrome.storage.local.set({ userId: crypto.randomUUID() });
        }
    });
});

let dwellTimer = null;
const DWELL_TIME_MS = 10000; // 10秒

chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
    if (changeInfo.status === 'complete' && tab.url) {
        checkUrlAndStartTimer(tab.url);
    }
});

async function checkUrlAndStartTimer(url) {
    try {
        const urlObj = new URL(url);
        let engine = null;
        let query = '';

        if (urlObj.hostname.includes('bing.com') && urlObj.searchParams.has('q')) {
            query = urlObj.searchParams.get('q');
            engine = 'bing';
        } else if (urlObj.hostname.includes('yahoo.co.jp') && urlObj.searchParams.has('p')) {
            query = urlObj.searchParams.get('p');
            engine = 'yahoo';
        }

        if (!engine || !query) return;

        // サーバーからタスク一覧を取得
        const res = await fetch(`${API_BASE}/daily`);
        const data = await res.json();
        const tasks = data.tasks || [];

        // 検索クエリの正規化
        const normalizedQuery = decodeURIComponent(query).toLowerCase().replace(/\s+/g, '');
        
        // 合致するタスクを探す
        const matchedTask = tasks.find(t => 
            t.engine === engine && normalizedQuery.includes(t.word_norm)
        );

        if (matchedTask) {
            console.log(`★タスク一致: ${matchedTask.word}`);
            if (dwellTimer) clearTimeout(dwellTimer);

            dwellTimer = setTimeout(() => {
                reportSuccess(matchedTask);
            }, DWELL_TIME_MS);
        }

    } catch (e) {
        console.error(e);
    }
}

// ポイント獲得報告
async function reportSuccess(task) {
    try {
        // ★ここを修正：メアド(userEmail)も一緒に取得して送る
        const { userId, userEmail } = await chrome.storage.local.get(['userId', 'userEmail']);
        
        if (!userId) return;

        const res = await fetch(`${API_BASE}/proof`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            // ★ここを修正：emailを追加
            body: JSON.stringify({ 
                userId: userId, 
                taskId: task.id,
                email: userEmail // これでサーバーのDBに記録されます
            })
        });
        const result = await res.json();
        
        if (result.success && result.awarded) {
            chrome.action.setBadgeText({ text: 'GET!' });
            chrome.action.setBadgeBackgroundColor({ color: '#F59E0B' });

            // 履歴保存
            chrome.storage.local.get(['pointHistory'], (res) => {
                const history = res.pointHistory || [];
                const now = new Date();
                const timeStr = `${now.getMonth()+1}/${now.getDate()} ${now.getHours()}:${now.getMinutes().toString().padStart(2, '0')}`;
                
                history.push({
                    date: timeStr,
                    word: task.word,
                    engine: task.engine,
                    pts: task.reward_points
                });
                chrome.storage.local.set({ pointHistory: history });
            });
        }
    } catch (e) {
        console.error(e);
    }
}