const API_BASE = "https://api.point.monster/api";

document.addEventListener('DOMContentLoaded', async () => {
    
    // ===============================================
    // 1. ログインチェック & 認証ロジック
    // ===============================================
    
    const { userId } = await chrome.storage.local.get(['userId']);

    if (userId) {
        showAppScreen(userId);
    } else {
        const loginView = document.getElementById('login-view');
        loginView.classList.remove('hidden');

        document.getElementById('login-btn').addEventListener('click', () => {
            const btn = document.getElementById('login-btn');
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Google認証中...';

            chrome.identity.getAuthToken({ interactive: true }, function(token) {
                if (chrome.runtime.lastError || !token) {
                    console.error(chrome.runtime.lastError);
                    alert('ログインできませんでした。');
                    btn.innerHTML = originalText;
                    return;
                }

                fetch('https://www.googleapis.com/oauth2/v2/userinfo', {
                    headers: { 'Authorization': 'Bearer ' + token }
                })
                .then(response => response.json())
                .then(async (userInfo) => {
                    await chrome.storage.local.set({ 
                        userId: userInfo.id, 
                        userEmail: userInfo.email 
                    });

                    // 設定画面用更新
                    if(document.getElementById('settings-email')) {
                         document.getElementById('settings-email').innerText = userInfo.email;
                         document.getElementById('settings-id').innerText = `ID: ${userInfo.id}`;
                    }

                    loginView.classList.add('hidden');
                    showAppScreen(userInfo.id);
                })
                .catch(err => {
                    console.error(err);
                    alert('ユーザー情報の取得に失敗しました');
                    btn.innerHTML = originalText;
                });
            });
        });
    }

    // ===============================================
    // 2. アプリメイン画面の処理
    // ===============================================
    
    async function showAppScreen(userId) {
        document.getElementById('app-view').classList.remove('hidden');
        
        // 設定画面用にメアドを表示（ストレージから再取得）
        chrome.storage.local.get(['userEmail'], (res) => {
            if(res.userEmail) {
                document.getElementById('settings-email').innerText = res.userEmail;
                document.getElementById('settings-id').innerText = `ID: ${userId}`;
            }
        });

        try {
            const [dailyRes, userRes] = await Promise.all([
                fetch(`${API_BASE}/daily`),
                fetch(`${API_BASE}/user?userId=${userId}`)
            ]);

            const dailyData = await dailyRes.json();
            const userData = await userRes.json();
            
            if (userData.status === 'blocked') {
                alert('このアカウントは利用が停止されています。');
                await logout();
                return;
            }

            const completedIds = userData.completedTaskIds || [];
            document.getElementById('date-label').innerText = `${dailyData.date.slice(5).replace('-','/')} のミッション`;
            document.getElementById('total-points').innerText = userData.balance.toLocaleString();

            const container = document.getElementById('task-list');
            container.innerHTML = '';

            if (!dailyData.tasks || dailyData.tasks.length === 0) {
                container.innerHTML = '<p style="text-align:center; color:#94a3b8; margin-top:20px;">今日のタスクはありません</p>';
                return;
            }

            // ★ここを変更：未完了を上に、完了済みを下にソートする
            dailyData.tasks.sort((a, b) => {
                const isA = completedIds.includes(a.id);
                const isB = completedIds.includes(b.id);
                // Aが完了でBが未完了なら、Aを後ろ(+1)にする
                if (isA && !isB) return 1;
                // Aが未完了でBが完了なら、Aを前(-1)にする
                if (!isA && isB) return -1;
                return 0; // どちらも同じなら順番変えない
            });

            dailyData.tasks.forEach(task => {
                const isCompleted = completedIds.includes(task.id);
                const engineName = task.engine === 'bing' ? 'Bing' : 'Yahoo';
                const iconClass = task.engine === 'bing' ? 'fa-brands fa-microsoft' : 'fa-brands fa-yahoo';
                const bgClass = task.engine === 'bing' ? 'bg-bing' : 'bg-yahoo';
                const topUrl = task.engine === 'bing' ? 'https://www.bing.com/' : 'https://www.yahoo.co.jp/';

                const btn = document.createElement('button');
                btn.className = `task-btn ${isCompleted ? 'completed' : ''}`;
                
                btn.innerHTML = `
                    <div class="engine-icon-area ${bgClass}">
                        <i class="${iconClass}"></i>
                    </div>
                    <div class="task-content">
                        <div class="task-word">${task.word}</div>
                        <div class="task-sub">${engineName}で検索を探す</div>
                    </div>
                    <div class="task-reward">
                        ${isCompleted ? '<i class="fa-solid fa-check"></i>' : `+${task.reward_points}pt`}
                    </div>
                `;

                if (!isCompleted) {
                    btn.addEventListener('click', () => {
                        navigator.clipboard.writeText(task.word);
                        
                        const subText = btn.querySelector('.task-sub');
                        const originalText = subText.innerText;
                        subText.innerText = 'コピーしました！';
                        subText.style.color = '#f59e0b';
                        subText.style.fontWeight = '900';

                        setTimeout(() => {
                            chrome.tabs.create({ url: topUrl });
                            subText.innerText = originalText;
                            subText.style.color = '#94a3b8';
                            subText.style.fontWeight = 'bold';
                        }, 1500);
                    });
                }
                container.appendChild(btn);
            });

        } catch (e) {
            console.error("データ取得エラー", e);
        }
    }

    // ===============================================
    // 3. その他（履歴・ログアウト）
    // ===============================================

    // 画面切り替えユーティリティ
    function hideAllViews() {
        document.getElementById('main-view').style.display = 'none';
        document.getElementById('history-view').style.display = 'none';
        document.getElementById('settings-view').style.display = 'none';
    }

    // 履歴画面へ
    document.getElementById('history-link').addEventListener('click', () => {
        hideAllViews();
        document.getElementById('history-view').style.display = 'block';
        loadHistory();
    });

    // 設定画面へ
    document.getElementById('settings-link').addEventListener('click', () => {
        hideAllViews();
        document.getElementById('settings-view').style.display = 'block';
    });

    // 戻るボタン共通処理
    document.querySelectorAll('.back-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            hideAllViews();
            document.getElementById('main-view').style.display = 'flex';
        });
    });

    // ログアウト処理
    document.getElementById('logout-btn').addEventListener('click', async () => {
        if(confirm('ログアウトしますか？')) {
            await logout();
        }
    });

    async function logout() {
        chrome.identity.getAuthToken({ interactive: false }, function(token) {
            if (token) {
                chrome.identity.removeCachedAuthToken({ token: token }, function() {});
            }
        });
        await chrome.storage.local.clear();
        window.location.reload();
    }

    async function loadHistory() {
        const { pointHistory } = await chrome.storage.local.get('pointHistory');
        const list = document.getElementById('history-list');
        list.innerHTML = '';
        if (!pointHistory || pointHistory.length === 0) {
            list.innerHTML = '<li style="text-align:center; color:#94a3b8; padding:20px;">履歴はありません</li>';
            return;
        }
        pointHistory.slice().reverse().forEach(item => {
            const listEl = document.createElement('li');
            listEl.className = 'history-item';
            listEl.innerHTML = `<div>${item.date} <span style="font-weight:bold;">${item.word}</span></div><div style="font-weight:900; color:#f97316;">+${item.pts}pt</div>`;
            list.appendChild(listEl);
        });
    }
});